/************************************************************
* MrsBot by Hendrix <jimi@rahul.net>                        *
* ban.c                                                     *
*   Handles banning of users and clearing of bans           *
*   Contains handler for numeric 302 (QUOTE USERHOST reply) *
*    which is also used for protect and unprotect commands. *
* Includes routines:                                        *
*   void onuserhost                                         *
*   void domegaban                                          *
*   void clearbans                                          *
*   void onbanlist                                          *
*   void onendofbans                                        *
*   void load_adios                                         *
*   struct adios_rec *parseadiosline                        *
*   void bldadiossub                                        *
*   void doadios                                            *
************************************************************/

#define MAXFUNNYKICKS 10
#define SUBST(x) (tmpptr->substitutions[x] == SUB_CHANNEL ? mychannel : \
		 (tmpptr->substitutions[x] == SUB_KICKER ? requestor : \
		 (tmpptr->substitutions[x] == SUB_RANDOM ? randomuser() : \
		 (tmpptr->substitutions[x] == SUB_KICKEE ? nick : NULL))))

/* Definitions for ADIOS kicks: both actions and substitution strings */
#define AD_KICK 0
#define AD_BAN 1
#define AD_MEGA 2
#define AD_SAY 3
#define AD_ACTION 4
#define SUB_NONE 0
#define SUB_CHANNEL 1
#define SUB_KICKER 2
#define SUB_KICKEE 3
#define SUB_RANDOM 4

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "config.h"

extern char mychannel[80]; /* MrsBot's current channel */
extern char buff[255];     /* Used to hold msgs to server */
extern char *randomuser();

void domegaban (char *userhost);
void clearbans(char* matching);
struct adios_rec *parseadiosline(char* line);
void doadios(int which,char* nick,char* userhost,char* requestor);
void bldadiossub(char *line, struct adios_rec *arec);
void douhaction(char* nick,char* userhost);

struct requestinfo uhostinfo; /* Info used by QUOTE USERHOST reply */
char waitingbans[255];     /* Holds bans queued for removal 3 at a time */
char waitingmode[5];       /* Holds mode string ("-bbb") for waitingbans */
char banstoclear[80];      /* Ban mask currently being unbanned */
char unbaninprogress;      /* 0 if unban going on, 1 if no unban */

struct adios_rec {
  char action;
  char *speakstr;
  char substitutions[5];
  struct adios_rec *next;
  };

int funnykickcnt = 0;
struct adios_rec *funnykicks[MAXFUNNYKICKS];

/*
** onuserhost()
**   Handles server reply from USERHOST msg (numeric 302)
**   Parameters:
**     reply - Numeric return info (e.g. "Hendrix=+jimi@netcom.netcom.com")
**   Returns: void
**   PDL:
**     Parse the nickname as the part before the =.  If there is no nickname
**     part, the nickname was not found, so send an error notice to the
**     person who requested the last command.  Otherwise, get the userhost from
**     the part two chars after the =.  Then handle the appropriate action.
**     ADDED: IRCops are returned with a '*' appended to their nickname, take
**     it off if it is there.  Ops are so fun to ban! :)
*/
void onuserhost (reply)
char *reply;
{
  char *nick, *userhost;
  char msgstr[80];

  nick = strtok(reply,"=");
  if (!nick) {
    sprintf(msgstr,"Nickname %s is not logged on",uhostinfo.targetnick);
    notice(uhostinfo.req_nick,msgstr);
    return;
    }
  if (nick[strlen(nick)-1] == '*')
    nick[strlen(nick)-1] = '\0';
  userhost = strtok(NULL," \0");
  if (userhost) {
    ++userhost;   /* Skip the first character in the userhost return string */
    douhaction(nick, userhost);
    }
}

/*
** douhaction()
**   Does the appropriate action upon the successful location of a
**    userhost, either from the server or via MrsBot's internal store.
**   Parameters:
**     nick - Nickname located
**     userhost - Userhost located
**   Returns: void
**   PDL:
**     Check uhostaction for what to do with this nick & userhost...
**     Handle MEGABAN, UNBAN, ADIOS (kick & ban), BAN, PROTECT, and
**     UNPROTECT.  Call protecteduh to make sure we aren't banning a
**     protected user.  V2 & V2.5 MODS:  Handle SETUSER and SHOWACCESS
**     situations.  Global uhostinfo holds information about the current
**     USERHOST request, with extrainfo being used in ADIOS and SETUSER.
*/
void douhaction(char* nick,char* userhost)
{
  char unbanstr[80];
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else
     strcpy(channel, mychannel);
     
  switch (uhostinfo.uhostaction) {
    case MEGABAN:
      if (!protecteduh(userhost))
        domegaban (userhost);
      else {
        sprintf(unbanstr,"Cannot ban %s: User is protected",
                uhostinfo.targetnick);
        notice(uhostinfo.req_nick,unbanstr);
        }
      break;
    case UNBAN:
      sprintf (unbanstr,"%s!%s", nick, userhost);
      clearbans (unbanstr);
      break;
    case ADIOS:
      if (!protecteduh(userhost))
        doadios(uhostinfo.extrainfo, nick, userhost, uhostinfo.req_nick);
      else {
        sprintf(unbanstr,"Cannot ADIOS %s: User is protected",
                uhostinfo.targetnick);
        notice(uhostinfo.req_nick,unbanstr);
        }
      break;
    case NORMALBAN:
      if (!protecteduh(userhost)) {
        sprintf (buff,"MODE %s +bb %s!*@* *!%s\n",channel,nick,userhost);
        toserv (buff);
        }
      else {
        sprintf(unbanstr,"Cannot ban %s: User is protected",
                uhostinfo.targetnick);
        notice(uhostinfo.req_nick,unbanstr);
        }
      break;
    case PROTECT:
      addprotect(userhost,uhostinfo.req_nick,nick);
      break;
    case UNPROTECT:
      rmprotect(userhost,uhostinfo.req_nick,nick);
      break;
    case SETUSER:
      changeuser(userhost,uhostinfo.extrainfo,0,uhostinfo.req_nick,nick);
      break;
    case SHOWACCESS:
      showaccess(nick,userhost,uhostinfo.req_nick);
      break;
    }
}

/*
** domegaban()
**   Bans a user with 2 different alternating ? strings on their userhost
**   Parameters:
**     userhost   - user@host to ban
**   Returns: void
**   PDL:
**     Build a ban command for the userhost twice.  Put an extra newline
**     between the two bans to help locate where the second ban begins.
**     Get the location of the second ban and change the newline to a
**     space like it should be.  Go thru each ban string's userhost and
**     change alternate characters in each to ?'s.  Send the ban command
**     to the server.  Note that for V2.5 this actually DOES the banning.
*/
void domegaban (char *userhost)
{
  int flipflop = 0;
  int i;
  char *ban1,*ban2;
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else
     strcpy(channel, mychannel);

  sprintf(buff,"MODE %s +bb *!%s *!%s\n",channel,userhost,userhost);
  ban1 = buff + 12 + strlen(channel);
  ban2 = ban1 + 3 + strlen(userhost);
  for (i=0; i<strlen(userhost); ++i,++ban1,++ban2)
    if (*ban1 != '@') {
      if (flipflop = !flipflop)
        *ban1 = '?';
      else
        *ban2 = '?';
      }
  toserv(buff);
}

/*
** clearbans()
**   Removes all bans which match the given nick!user@host string.
**   Or removes ALL bans if an empty string is given.
**   Parameters:
**     matching - nick!user@host to unban or "" for all bans
**   Returns: void
**   PDL:
**     Mark that an unban is in progress.  Copy the nick!user@host
**     into a global for later, then request a list of all bans.
*/
void clearbans(char* matching)
{
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else
     strcpy(channel, mychannel);
  unbaninprogress = 1;
  strcpy(banstoclear,matching);
  sprintf (buff,"MODE %s +b\n", channel);
  toserv (buff);
  *waitingmode = *waitingbans = '\0';
}

/*
** onbanlist()
**   Handles msgs coming from a listing of the ban list (numeric 367)
**   Or removes ALL bans if an empty string is given.
**   Parameters:
**     item - a single nick!user@host mask from the ban list
**   Returns: void
**   PDL:
**     If we are clearing all bans or if this ban-mask matches the
**     nick!user@host we are unbanning, check the length of the
**     ban string we have pending.  If the ban string is over 78
**     characters long, or if we already are unbanning 4 users in
**     a single command, send the unban request to the server.
**     Add the ban mask onto the list of bans to be removed, and
**     add one more "b" onto the "/mode * -b" command.
*/
void onbanlist(item)
char *item;
{
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else
     strcpy(channel, mychannel);
  if (*banstoclear == '\0' || !wldcmp(item,banstoclear)) {
    if ((strlen(waitingbans) + strlen(item) > 78) || strlen(waitingmode) > 3) {
      sprintf (buff,"MODE %s -%s%s\n", channel, waitingmode, waitingbans);
      toserv (buff);
      *waitingmode = *waitingbans = '\0';
      }
    strcat (waitingmode,"b");
    strcat (waitingbans," ");
    strcat (waitingbans,item);
    }
}

/*
** onendofbans()
**   Handles the end of ban list msg (numeric 368)
**   Parameters: None
**   Returns: void
**   PDL:
**     If we have an unfinished unban command waiting to go to the
**     server, send it.  Then mark the unban as complete.
*/
void onendofbans()
{
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else
     strcpy(channel, mychannel);
  if (*waitingbans) {
    sprintf (buff,"MODE %s -%s%s\n", channel, waitingmode, waitingbans);
    toserv (buff);
    }
  unbaninprogress = 0;
}

/*
** load_adios()
**   Initialize the list of "funny" kick sequences for ADIOS.
**   Parameters: None.
**   Returns: void
**   PDL:
**     Open file adios.load and read in one line at a time.  If
**     the line begins with a #, it marks the end of an ADIOS
**     sequence.  In that case, mark the end of the current linked
**     list of ADIOS actions with a NULL, then increment the count
**     of funny kicks.  If the line doesn't start with a #, call
**     parseadiosline() to create a linked list node that describes
**     this action in the sequence.  Attach it to the end of the
**     linked list of actions for this funny kick, and mark the new
**     last item in the linked list.
*/
void load_adios()
{
  FILE *infile;
  char line[255];
  struct adios_rec *currec,*lastrec = NULL;

  infile = fopen("adios.load","r");
  while (fgets(line, 255, infile) != NULL) {
    line[strlen(line)-1] = '\0';
    if (*line == '#') {
      if (lastrec) {
	lastrec->next = NULL;
	lastrec = NULL;
	if (++funnykickcnt == MAXFUNNYKICKS)
	  break;
	}
      }
    else {
      currec = parseadiosline(line);
      if (currec) {
	if (lastrec)
	  lastrec->next = currec;
        else
	  funnykicks[funnykickcnt] = currec;
	lastrec = currec;
	}
      }
    }
  fclose (infile);
}

/*
** parseadiosline()
**   Converts a line from the adios.load file into a record that
**    describes the action to perform during an ADIOS.
**   Parameters:
**     line - A line from the adios.load file
**   Returns:
**     A pointer to a record representing this line
**   PDL:
**     Look at the first character in the line and determine the
**     type of action that is represented: for A & S, set the
**     record flag to indicate an ACTION or a SAY command, then
**     parse the rest of the line for what to do or say.  The
**     others (K,B,?) are pretty obvious and the rest of the
**     line is ignored.
*/
struct adios_rec *parseadiosline(char* line)
{
  struct adios_rec *temp;

  temp = malloc(sizeof (struct adios_rec));
  switch (*line) {
    case 'A':
      temp->action = AD_ACTION;
      bldadiossub(&line[1],temp);
      break;
    case 'S':
      temp->action = AD_SAY;
      bldadiossub(&line[1],temp);
      break;
    case 'K':
      temp->action = AD_KICK;
      break;
    case 'B':
      temp->action = AD_BAN;
      break;
    case '?':
      temp->action = AD_MEGA;
      break;
    default:
      free (temp);
      return NULL;
      }
  return temp;
}

/*
** bldadiossub()
**   Takes a user string for an action or say string in an ADIOS sequence
**    and converts it to the format stored in an adios_rec.
**   Parameters:
**     line - String from adios.load file to do or say.
**     arec - adios_rec to be filled in
**   Returns: void
**   PDL:
**     Very similar to parsehelpercmd().  Scan through the string passed in,
**     copying letter for letter, unless a $ or % is found.  The first five
**     $K,$M,$R, or $C strings found are considered substitutions.  Future
**     instances of these are ignored.  For each of the above strings, place
**     a %s in the template string (for sprintf use) and set the next
**     substitution field to the type of item to be substituted (kickee,
**     kicker, random, or channel).  % literals in the string need to be
**     escaped into %% for proper functioning in sprintf.  When the entire
**     string has been copied, fill in any unused substitution fields with
**     NULL, then copy the template into the adios_rec.
*/
void bldadiossub(char *line, struct adios_rec *arec)
{
  int subcnt = 0;
  char hold[200];
  char *template = hold;

  while (*line) {
    if (*line == '$')
      if (subcnt < 5) {
        switch (*(++line)) {
          case 'K':
	  case 'k':
	    arec->substitutions[subcnt] = SUB_KICKEE;
	    break;
          case 'M':
          case 'm':
	    arec->substitutions[subcnt] = SUB_KICKER;
	    break;
          case 'R':
          case 'r':
	    arec->substitutions[subcnt] = SUB_RANDOM;
	    break;
          case 'C':
          case 'c':
	    arec->substitutions[subcnt] = SUB_CHANNEL;
	    break;
          default:
	    arec->substitutions[subcnt] = SUB_NONE;
	    *(template++) = *line;
	    break;
          }

	if (arec->substitutions[subcnt]) {
	  *(template++) = '%';
	  *(template++) = 's';
	  subcnt++;
	  }
        }
      else
	*(template++) = *(++line);
    else if (*line == '%') {
      *(template++) = '%';
      *(template++) = '%';
      }
    else
      *(template++) = *line;
    ++line;
    }
  while (subcnt < 5)
    arec->substitutions[subcnt++] = SUB_NONE;
  *template = '\0';

  arec->speakstr = (char *)malloc(strlen(hold)+1);
  strcpy(arec->speakstr,hold);
}

/*
** doadios()
**   Performs a sequence of actions for an ADIOS
**   Parameters:
**     which     - The number of the funny kick sequence to perform
**     nick      - The nickname that will get kicked and banned
**     userhost  - The userhost that will be banned
**     requestor - The nick that requested this ADIOS action
**   Returns: void
**   PDL:
**     Pretty intuitive looking code... Go thru the linked list of adios
**     action records for this particular sequence.  For each one, do what
**     it says: kick, ban, megaban, say something, or do a CTCP ACTION.
**     For SAY and ACTION, use sprintf to build a string containing the
**     proper fields from the template.  Macro SUBST (at the top of this
**     file) does most of the hard work there.
*/
void doadios(int which,char* nick,char* userhost,char* requestor)
{
  struct adios_rec *tmpptr;
  char saystr[200];
  char channel[80];
  if (mychannel[0] == ':')
     strcpy(channel, mychannel+1);
  else 
     strcpy(channel, mychannel);
  
  tmpptr = funnykicks[which];
  while (tmpptr) {
    switch (tmpptr->action) {
      case AD_KICK:
        kick(channel,nick);
	break;
      case AD_BAN:
        sprintf (buff,"MODE %s +bb %s!*@* *!%s\n", channel, nick,
		 userhost);
        toserv (buff);
	break;
      case AD_MEGA:
        domegaban (userhost);
	break;
      case AD_SAY:
        sprintf (saystr, tmpptr->speakstr, SUBST(0), SUBST(1), SUBST(2),
	         SUBST(3), SUBST(4));
	say (saystr);
	break;
      case AD_ACTION:
        sprintf (saystr, tmpptr->speakstr, SUBST(0), SUBST(1), SUBST(2),
	         SUBST(3), SUBST(4));
	action (saystr);
	break;
      }
    tmpptr = tmpptr->next;
    }
}
