/************************************************************
* MrsBot by Hendrix <jimi@rahul.net>                        *
* beatban.c                                                 *
* Handles getting back on to a channel when the             *
*  channel is mode +i or MrsBot is banned.                  *
* Also handles list of helper bots and requesting of ops    *
*  from these bots.                                         *
* Includes routines:                                        *
*   void whenbanned                                         *
*   void wheninvonly                                        *
*   void whenchanfull                                       *
*   void onnames                                            *
*   void onendofnames                                       *
*   void askforops                                          *
*   char ishelperbot                                        *
*   void onwhoischans                                       *
*   void onnosuchnick                                       *
*   void load_helpers                                       *
*   void parse_helpercmd                                    *
************************************************************/

#define MAX_HELPER_BOTS 10

/* Allowable values for substitution strings */
#define NO_SUBSTITUTION 0
#define INSERT_NICK 1
#define INSERT_CHANNEL 2

/* Cute macro used for nick/channel substitution into a string */
#define SUBST(x,y) (helpers[x].substitutions[y] == INSERT_NICK ? mynick : \
		   (helpers[x].substitutions[y] == INSERT_CHANNEL ? \
		    chantojoin : "" ))

#include <string.h>
#include <stdio.h>
#include "config.h"

extern char wewerebanned;

extern void beatban(char* User);
void load_helpers();
void onnosuchnick(char* nick);
void onwhoischans(char* chanline);
char ishelperbot(char* nick);
void askforops();
void onendofnames();
void onnames(char* nameline);
void whenchanfull(char* channel);
void wheninvonly(char* channel);
void whenbanned(char* channel);
void parse_helpercmd (char* template, char* item_to_sub);

extern char mynick[10];    /* MrsBot's current nickname */
extern FILE* outfile;
char helpernum = 0;        /* Holds next helper to be asked to help
		              MrsBot rejoin a channel */
char whatneeded = CAN_JOIN;/* What we need to rejoin the channel
			      (invite,unban,etc.) */
int bantest = 1;
char chantojoin[80];       /* channel we are trying to rejoin */
char *strfromfile;         /* Current string parsed from init file */

struct helper_rec {
  char nick[10];
  char inv_str[20];
  char deban_str[20];
  char op_str[20];
  char substitutions[3];
  };

/* List of bots/people who will invite, unban, and op us,
   and the command strings that will request each. */
struct helper_rec helpers[MAX_HELPER_BOTS];

/*
** whenbanned()
**   Called in response to a banned from channel msg (numeric 474)
**   Parameters:
**     channel - Channel from which we are banned
**   Returns: void
**   PDL:
**     The first time we detect we are banned, change to a random nickname
**     and attempt to rejoin the channel.  If that fails, change back to
**     the normal nickname, and see if we have begun attempts to rejoin
**     the channel already.  (whatneeded == CAN_JOIN if we haven't tried
**     to rejoin the channel yet.)  If we have NOT begun attempts to
**     rejoin the channel, initialize the globals to start looking for
**     helper bots from the beginning of the list.  If we have, advance
**     to looking for the next helper bot on the list.  If we go thru
**     the entire list, schedule another join attempt for 2 minutes later.
**     If we still have bots to check for, do a WHOIS on that bot name to
**     see if it is opped on the channel.
*/
void whenbanned(char* channel)
{
    static int count = 0;
	 if (!count)
	 {
		 wewerebanned = 1;
		 randnick();
		 newnick(mynick);
		 count++;
       join(channel);
    }
	 else if (count < 5)
	 {
		 wewerebanned = 1;
       beatban(NULL); 
       count++;
       join(channel);
    }
	 else
	 {
		 msg(MASTER, "Unban me\n");
		 join(channel);
       count = 0;
    }
}

/*
** wheninvonly()
**   Called in response to a need invite msg (numeric 473)
**   Parameters:
**     channel - Channel from which we are banned
**   Returns: void
**   PDL:
**     Similar to whenbanned(), see if we have begun attempts to rejoin
**     the channel already.  (whatneeded == CAN_JOIN if we haven't tried
**     to rejoin the channel yet.)  If we have NOT begun attempts to
**     rejoin the channel, initialize the globals to start looking for
**     helper bots from the beginning of the list.  If we have, advance
**     to looking for the next helper bot on the list.  If we go thru
**     the entire list, schedule another join attempt for 2 minutes later.
**     If we still have bots to check for, do a WHOIS on that bot name to
**     see if it is opped on the channel.
*/
void wheninvonly(char* channel)
{
  if (whatneeded == CAN_JOIN) {
    whatneeded = WANT_INVITE;
    helpernum = 0;
    }
  else if (whatneeded == WANT_INVITE)
    ++helpernum;
  if (*helpers[helpernum].nick) {
    strcpy(chantojoin,channel);
    whois (helpers[helpernum].nick);
    }
  else
    timedjoin(120,channel);
}

/*
** whenchanfull()
**   Called in response to a too many users on channel msg (numeric 471)
**   Parameters:
**     channel - Channel from which we are banned
**   Returns: void
**   PDL:
**     This sucks... but honestly how many times are you locked out of
**     a channel cuz of a mode +l?
*/
void whenchanfull(char* channel)
{
  msg(MASTER, "Mode -l please!");
  timedjoin(60,channel);
}

/*
** onwhoischans()
**   Handles a single line of the WHOIS reply, namely the one holding
**    the channels that the user is on. (numeric 319)
**   Parameters:
**     chanline - The full line of reply from the server
**   Returns: void
**   PDL:
**     Since this message is returned by the server when MrsBot does a
**     WHOIS on a helper bot, search the list of channels that the bot
**     is on for the channel which MrsBot is trying to rejoin.  Prepend
**     a @ to the channel name to serach for to make sure that the
**     helper bot is opped and can help.  If it is found, build a msg
**     for either unbanning or an invitation and send it to the bot.
**     Then time an attempt to rejoin the channel in 20 seconds, to
**     make sure the helper bot has enough time to process the msg.
**     Otherwise, proceed on to the next helper in the list.  If we
**     have not exhausted all helper bots, send a WHOIS on the next
**     bot in the list.  If we have, just time another rejoin for 2
**     minutes later and hope that the ban/invite is gone by then.
*/
void onwhoischans(char* chanline)
{
  char chanpattern[80];
  char msgstr[80];

  chanpattern[0] = '@';
  strcpy(chanpattern+1,chantojoin);
  if (strstr(chanline,chanpattern)) {
    if (whatneeded == WANT_INVITE)
      sprintf(msgstr,helpers[helpernum].inv_str, SUBST(helpernum,0));
    else
      sprintf(msgstr,helpers[helpernum].deban_str, SUBST(helpernum,1));
    msg(helpers[helpernum].nick,msgstr);
    timedjoin(20,chantojoin);
    }
  else {
    ++helpernum;
    if (*helpers[helpernum].nick)
      whois(helpers[helpernum].nick);
    else
      timedjoin(120,chantojoin);
    }
}

/*
** onnosuchnick()
**   Handles a response that a given nickname does not exist.
**   Parameters:
**     nick - nickname that could not be found by the server
**   Returns: void
**   PDL:
**     This message is hopefully only returned by the server when
**     MrsBot does a WHOIS on a helper bot that is not currently
**     logged on.  It can, in fact, also occur for various other
**     reasons like PRIVMSGs and NOTICEs to nicks that have since
**     logged off.  First, verify that we are indeed in the process
**     of trying to rejoin a channel, then also verify that the
**     nickname that did not exist was the last helper bot that we
**     did a WHOIS on.  This being the case, proceed on to the next
**     helper in the list.  If we have not exhausted all helper
**     bots, send a WHOIS on the next bot in the list.  If we have,
**     just time another rejoin for 2 minutes later and hope that
**     the ban/invite is gone by then.
*/
void onnosuchnick(char* nick)
{
  if (whatneeded != CAN_JOIN && !strcasecmp(helpers[helpernum].nick,nick)) {
    ++helpernum;
    if (*helpers[helpernum].nick)
      whois(helpers[helpernum].nick);
    else
      timedjoin(120,chantojoin);
    }
}

/*
** onnames()
**   Handles a single line of a reply from a NAMES request (numeric 353)
**   Parameters:
**     nameline - The full line of reply from the server
**   Returns: void
**   PDL:
**     In version 1, this was used to locate a helper bot on a channel.
**     In version 2, NAMES has been replaced with WHOIS, so this remains
**     but as a null function.
*/
void onnames(char* nameline)
{
}

/*
** onendofnames()
**  Handles message indicating that there are no more names to list for
**    the channel.  (numeric 366)
**   Parameters: None.
**   Returns: void
**   PDL:
**     In version 1, this was used to locate a helper bot on a channel.
**     In version 2, NAMES has been replaced with WHOIS, so this remains
**     but as a null function.
*/
void onendofnames()
{
}

/*
** askforops()
**   Called after a period of time expires after the bot has joined
**    the channel or been deopped, and it STILL has not been opped.
**   Parameters: None.
**   Returns: void
**   PDL:
**     Since we are on a channel when this is called by definition,
**     check MrsBot's internal store of channel users to see if any
**     helper bots are opped and on the channel, and if the helper
**     has an "ask for op" string defined for it.  If it passes the
**     above, build the "ask for op" string and say it on the channel.
*/
void askforops()
{
  int i = -1;
  char buff[80];

  while (*helpers[++i].nick)
    if (*helpers[i].op_str && isnickopped(helpers[i].nick)) {
      sprintf(buff,helpers[i].op_str,SUBST(i,2));
      say (buff);
      break;
      }
}

/*
** ishelperbot()
**   Called to determine if given nickname is a helper bot that
**    MrsBot is interested in keeping track of.
**   Parameters:
**     nick - Nickname of user to check against list of helper bots
**   Returns: void
**   PDL:
**     Compare the nick with each bot on the helper bot list.  Easy, huh?
*/
char ishelperbot(char* nick)
{
  int i = -1;

  while (*helpers[++i].nick)
    if (!strcasecmp(helpers[i].nick,nick))
      return 1;
  return 0;
}

/*
** load_helpers()
**   Initialize the list of helper bots and the command strings
**    each uses for unban, invite, and op.
**   Parameters: None.
**   Returns: void
**   PDL:
**     Open file helpers.load and read in one helper bot per
**     line.  Parse the nick off the beginning of the string
**     and copy it into the next available entry in the helper
**     list.  Call parse_helpercmd() to take the next items off
**     the line: the invite, unban, and op commands.  If there
**     are more than MAX_HELPER_BOTS records, the list is
**     truncated at MAX_HELPER_BOTS.
*/
void load_helpers()
{
  FILE *infile;
  char line[255], *nick;
  int cnt = 0;

  infile = fopen("helpers.load","r");
  while (fgets(line, 255, infile) != NULL) {
    line[strlen(line)-1] = '\0';
    nick = strtok(line,"#");
    strcpy(helpers[cnt].nick,nick);

    strfromfile = strtok(NULL,"\0");
    parse_helpercmd(helpers[cnt].inv_str,&helpers[cnt].substitutions[0]);
    parse_helpercmd(helpers[cnt].deban_str,&helpers[cnt].substitutions[1]);
    parse_helpercmd(helpers[cnt].op_str,&helpers[cnt].substitutions[2]);
    if (++cnt == MAX_HELPER_BOTS - 1)
      break;
    }

    *helpers[cnt].nick = '\0';
    close(infile);
}

/*
** parse_helpercmd()
**   Takes a template for a command to a helper bot and converts it
**    into the format stored in the helper list.
**   Parameters:
**     template - Location to store "sprintf" format helper cmd string
**     item_to_sub - Location to mark type of item to be subbed into template
**   Returns: void
**   PDL:
**     Assume from the start that there are no strings to be substituted
**     into the command string.  Scan through the remainder of the last line
**     read in from the helper file looking for a # or the end of line, both
**     of which will mark the end of the command.  Copy the string letter for
**     letter into 'template' with the following special cases.  Convert
**     $C or $N into %s (for sprintf use) and set 'item_to_sub' to indicate
**     whether to use the nick or channel name ($N or $C, respectively) for
**     later substitution into this string.  Note that only one substitution
**     string is allowed per template, and future $N/$C's are ignored.  The
**     only other special case is '%' symbols, which must be escaped into %%
**     for proper functioning in sprintf.
*/
void parse_helpercmd (char* template, char* item_to_sub)
{
  *item_to_sub = NO_SUBSTITUTION;
  if (strfromfile) {
    while (*strfromfile && *strfromfile != '#') {
      if (*strfromfile == '$')
	switch (*(++strfromfile)) {
          case 'N':
	  case 'n':
	    if (!*item_to_sub) {
	      *(template++) = '%';
	      *(template++) = 's';
	      *item_to_sub = INSERT_NICK;
	      }
            ++strfromfile;
	    break;
          case 'C':
          case 'c':
	    if (!*item_to_sub) {
	      *(template++) = '%';
	      *(template++) = 's';
	      *item_to_sub = INSERT_CHANNEL;
	      }
            ++strfromfile;
	    break;
          default:
	    *(template++) = *(strfromfile++);
	    break;
          }
      else if (*strfromfile == '%') {
	*(template++) = '%';
	*(template++) = *(strfromfile++);
	}
      else
	*(template++) = *(strfromfile++);
      }
    ++strfromfile;
    }
  *template = '\0';
}
