/************************************************************
* MrsBot by Hendrix <jimi@rahul.net>                        *
* timer.c                                                   *
*   Contains complicated timer code based around "alarm"    *
*    system call.                                           *
*   Handles low-level timer work for timed operations such  *
*    as repeated attempts to rejoin, delayed ops, killing   *
*    BabyBot when it is not being used, etc.                *
* Includes routines:                                        *
*   struct trackrec *monitornick                            *
*   void addtotimeq                                         *
*   void timedop                                            *
*   void timedmiss                                          *
*   void timedjoin                                          *
*   void timedspite                                         *
*   void timednick                                          *
*   void timedgetops                                        *
*   void timedbabydie                                       *
*   void timersignal                                        *
*   struct trackrec *locatenick                             *
*   char uh_nick_protected                                  *
*   char isnickopped                                        *
*   void onnick                                             *
*   char *whoopped                                          *
*   char *whodeopped                                        *
*   void wholeft                                            *
*   void alarmoff                                           *
*   void onpart                                             *
*   void resyncoplist                                       *
*   void onwho                                              *
*   void onendofwho                                         *
*   void onnicktaken                                        *
*   char *int_userhost                                      *
*   char *randomuser                                        *
************************************************************/

#define MASKSIGS 0x5a
#define ISMASKED(x) (x ^= sigmsk)

#include <sys/time.h>
#include <string.h>
#include <strings.h>
#include <sys/signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <unistd.h>
#include <sys/types.h>
#include "config.h"

/* Blecch... I use this value to check if a record has been freed or not,
   see below in monitornick() and wholeft(). */
#define FREED_REC_KLUDGE 0xdeadbeef

/* Different operations that can be queued to the timer */
#define DELAY_OP 1
#define DELAY_RESPITE 2
#define DELAY_REJOIN 3
#define DELAY_MISSHENDY 4
#define DELAY_RENICK 5
#define DELAY_SUBBOTDIE 6
#define DELAY_GETOPS 7
#define NOOPTOOP "I don't have op to op you, sorry... :("

extern char mynick[10];    /* MrsBot's current nickname */
extern char mychannel[80]; /* MrsBot's current channel */
extern char spite;         /* 1 if revenge kicking is enabled */
extern char opstate;       /* 1 is MrsBot is currently opped */
extern FILE *outfile;

/* Definition for an item on the queue of pending timer operations. */
struct time_q_rec {
  char operation;
  char *oper_info;  /* pointer to whatever we need to do operation:
                        for DELAY_OP - pointer to user's trackrec
                            DELAY_REJOIN - pointer to channel to rejoin */
  time_t expiretime;
  struct time_q_rec *next;
  };

struct trackrec *monitornick (char* nick,char* userhost,char opped);
void onnick(char* oldnick,char* newnick);
char *randomuser();
void wholeft(char* nick,char* userhost);
void timedsay();
struct trackrec *locatenick(char* nick);
char *whoopped(char* nick);
char *whodeopped(char* nick);
char *int_userhost(char* nick);
void onnicktaken();
void onendofwho(char* what);
void onwho(char* wholine);
void resyncoplist(char* chan);
void onpart(char* nick);
char isnickopped(char* nick);
char uh_nick_protected(char* nick_pattern,char* userhost_pattern);
void addtotimeq(struct time_q_rec *rec, int waittime);
void timersignal(char* signalvec);
void timedbabydie(int when,int socket);
void timedgetops(int when);
void timednick(int when);
void timedjoin(int when,char* chan);
void timedspite(int when);
void timedop(char* nick,char* userhost);
void alarmoff();

struct time_q_rec *timequeue = NULL;  /* Queue of pending timer operations */
struct trackrec *thoseweop = NULL;    /* MrsBot's store of channel users */
int usersonchan = 0;       /* Counter for number of users in thoseweop */

/*
** monitornick()
**   Adds a user record to MrsBot's internal store of channel users.
**   This is only called for users who are either on the channel when
**   MrsBot joins or join the channel while she is on.
**   Parameters:
**     nick - Nickname of user to add to the store
**     userhost - Userhost of user to add to the store
**     opped - Is this user currently opped?
**   Returns:
**     Pointer to the trackrec that MrsBot has placed into the store.
**   PDL:
**     Pretty well documented in the code here.  If ya got questions,
**     track me down. :)  Just remember, the internal user store
**     (thoseweop) is not just people we op... and that it is just a
**     linked list of trackrec's.
*/
struct trackrec *monitornick (char* nick,char* userhost,char opped)
{
  struct trackrec *temprec,*tmpptr;

  /* THIS CODE BUILDS A RECORD FOR THIS USER FOR
     TRACKING CHANGES IN OP STATUS */
  temprec = malloc(sizeof (struct trackrec));

  /* THIS IS VERY UGLY.  PUT HEX "deadbeef" INTO A KEY.  THE TIME QUEUE
     CHECKS THIS KEY TO SEE IF THE TRACKREC IT POINTS TO HAS BEEN
     DEALLOCATED. */
  temprec->Kludge_Key = FREED_REC_KLUDGE;
  strcpy(temprec->nick,nick);
  strcpy(temprec->userhost,userhost);
  temprec->opped = opped;
  temprec->next = NULL;

  /* ADD RECORD TO LIST OF NICKS BEING MONITORED ON CHANNEL */
  if (thoseweop) {
    tmpptr = thoseweop;
    while (tmpptr->next)
      tmpptr = tmpptr->next;
    tmpptr->next = temprec;
    }
  else
    thoseweop = temprec;

  ++usersonchan;
  return temprec;
}

/* SEARCHES THE QUEUED OPERATIONS AND INSERTS THIS RECORD INTO THE
   LIST KEEPING IT IN ASCENDING ORDER BASED ON TIME OF EXECUTION.
   RESETS A TIMER IF NECESSARY TO EXPIRE ON THE FIRST ITEM OF THE
   LIST.  YES... I KNOW MY QUEUE OPERATIONS ARE NOT ATOMIC AND
   COULD FAIL IF THE SIG_ALRM HITS AT THE WRONG TIME.*/
/*
** addtotimeq()
**   Adds a timer record into the queue of pending timer operations.
**   Parameters:
**     rec - A previously built timer queue record for this item
**     waittime - Time in seconds to wait before perforiming this action
**   Returns: void
**   PDL:
**     Not as easy as it sounds... Because things are popped off the
**     timer queue as their timer expires, and because Unix is kind
**     enough to only allow one timer (alarm) request at a time, we
**     have to keep this list in time order, that is ascending order
**     based on the time that the command should be executed.  Now,
**     for the PDL.  Shut off any pending timers so they don't go off
**     in the middle of us messing with the timer queue.  Figure out
**     the actual time of execution for this command by adding the
**     waittime to the time now.  If there is nothing on the timer
**     queue, we got it easy: Put this one on the head.  Otherwise,
**     search down the list until we find an item that has a later
**     expire time than this record.  Insert the new record before
**     this record.  Then recalculate the amount of time before
**     the record at the TOP of the queue is to go off, and set an
**     alarm to wait this length of time.  NOTE: as timers expire
**     alarmoff() is called by Unix to handle them.
*/
void addtotimeq(rec,waittime)

struct time_q_rec *rec;
int waittime;
{
struct time_q_rec *tmpptr;
  /* NEWS FLASH: TIMER OPERATIONS ARE FINALLY PROTECTED FROM
     SIG_ALRM.  I GOT SICK OF UNEXPLAINED BUGS! :) */
  alarm (0);
  rec->expiretime = time(NULL)  + waittime;
  if (timequeue) {
    if (timequeue->expiretime < rec->expiretime) {
      tmpptr = timequeue;
      while (tmpptr->next)
        if (tmpptr->next->expiretime >= rec->expiretime)
          break;
        else
          tmpptr = tmpptr->next;
      rec->next = tmpptr->next;
      tmpptr->next = rec;
      waittime = time(NULL) - timequeue->expiretime;
      if (waittime < 1)
        waittime = 1;  
      alarm(waittime);
      return;
      }
    }
  rec->next = timequeue;
  timequeue=rec;
  alarm(waittime);
}

/*
** timedop()
**   Adds a delayed op for a user onto the timer queue.
**   Parameters:
**     nick - Nickname to op
**     userhost - Userhost to op
**   Returns: void
**   PDL:
**     Build a timer record for the "op" operation.  Add this user to
**     the internal store of channel users for tracking.  Calculate a
**     random time to wait before opping the user.  Queue the record
**     on the timer queue.
*/
void timedop(char* nick,char* userhost)
{
  struct time_q_rec *tempqrec;
  int waittime;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->oper_info = (char *)monitornick(nick,userhost,0);
  tempqrec->operation = DELAY_OP;

  /* WAIT 3-10 SECONDS BEFORE OPPING USER */
  waittime = 1;
  addtotimeq(tempqrec,waittime);
}

/*
** timedmiss()
**   Add a whine to the channel about missing Hendrix on the timequeue
**   Parameters:
**     when - Time to wait before whining to the channel
**   Returns: void
**   PDL:
**     Build a timer record for the "misshendy" operation.  Add this
**     record to the time queue.  Mark that Hendrix is gone, so that
**     we can test this flag to see if he is back before we whine.
*/

/*
** timedjoin()
**   Add a rejoin to the channel onto the timequeue.  Called after
**    failing to rejoin a channel.
**   Parameters:
**     when - Seconds to wait before attempting rejoin
**     chan - Channel name to rejoin
**   Returns: void
**   PDL:
**     Build a timer record for the "rejoin" operation.  Add this
**     record to the timer queue.
*/
void timedjoin(int when,char* chan)
{
  struct time_q_rec *tempqrec;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->oper_info = (char *)malloc(strlen(chan)+1);
  strcpy(tempqrec->oper_info,chan);
  tempqrec->operation = DELAY_REJOIN;
  addtotimeq(tempqrec,when);
}

/*
** timedspite()
**   Add a note to turn revenge kicking back on onto the timer queue.
**   Parameters:
**     when - Seconds to wait before turning revenge kicking back on
**   Returns: void
**   PDL:
**     Build a timer record for the "respite" operation.  Add this
**     record to the timer queue.
*/
void timedspite(int when)
{
  struct time_q_rec *tempqrec;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->operation = DELAY_RESPITE;
  addtotimeq(tempqrec,when);
}

/*
** timednick()
**   Add a note to attempt to change back to MrsBot's normal nickname
**    onto the timer queue.  Called when someone steals MrsBot's nick
**    and she needs to assume another one.
**   Parameters:
**     when - Seconds to wait before attempting to regain normal nick
**   Returns: void
**   PDL:
**     Build a timer record for the "renick" operation.  Add this
**     record to the timer queue.
*/
void timednick(int when)
{
  struct time_q_rec *tempqrec;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->operation = DELAY_RENICK;
  addtotimeq(tempqrec,when);
}

/*
** timedgetops()
**   Add a request to ask a helper bot for ops onto the timer queue.
**   Parameters:
**     when - Seconds to wait before asking a bot for ops
**   Returns: void
**   PDL:
**     Build a timer record for the "getops" operation.  Add this
**     record to the timer queue.
*/
void timedgetops(int when)
{
  struct time_q_rec *tempqrec;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->operation = DELAY_GETOPS;
  addtotimeq(tempqrec,when);
}

/*
** timedbabydie()
**   Add a request to kill off BabyBot if it is inactive to the timequeue
**   Parameters:
**     when - Seconds to wait before killing BabyBot
**   Returns: void
**   PDL:
**     Build a timer record for the "subbotdie" operation.  Add this
**     record to the timer queue.
*/
void timedbabydie(int when,int socket)
{
  struct time_q_rec *tempqrec;

  tempqrec = malloc(sizeof (struct time_q_rec));
  tempqrec->operation = DELAY_SUBBOTDIE;
  tempqrec->oper_info = (char *)socket;
  addtotimeq(tempqrec,when);
}

/*
** locatenick()
**   Locates a record for a given nickname in MrsBot's internal user store.
**   Parameters:
**     nick - Nickname to locate
**   Returns:
**     A pointer to the trackrec for this user, or NULL if not found.
**   PDL:
**     Scan the linked list of monitored users.  If we find a match before
**     the end of list, return this record, otherwise return the end of
**     list (NULL).
*/
struct trackrec *locatenick(char* nick)
{
  struct trackrec *tmpptr;

  tmpptr = thoseweop;
  while (tmpptr)
  {
    if (!strcmp(nick,tmpptr->nick))
      break;
    else
      tmpptr = tmpptr->next;
#ifdef DEBUGMODE
  /* fprintf(outfile, "Nick %s tmp'nick %s", nick, tmpptr->nick); */
#endif
   }
  return tmpptr;
}

/*
** uh_nick_protected()
**   Called when matching ban masks.  Determines if the given userhost
**    and nickname patterns match a protected user in MrsBot's internal
**    store.
**   Parameters:
**     nick_pattern - Wildcarded pattern for nickname to match
**     userhost_pattern - Wildcarded pattern for userhost to match.
**   Returns:
**     1 if a currently protected user matches both masks.
**   PDL:
**     Scan thru the list of monitored users.  When a user is found with
**     the userhost we are looking for, compare the nickname against the
**     given nick pattern.  If they match, and the user is protected,
**     return 1.  Otherwise, keep scanning through the list as there may
**     well be more than matching userhost/nickname combination.
*/
char uh_nick_protected(char* nick_pattern,char* userhost_pattern)
{
  struct trackrec *tmpptr;

  tmpptr = thoseweop;
  while (tmpptr) {
    if (!wldcmp(userhost_pattern,tmpptr->userhost))
      if (!wldcmp(nick_pattern,tmpptr->nick))
	if (protecteduh(tmpptr->userhost)) 
	  return 1;
    tmpptr = tmpptr->next;
    }
  return 0;
}

/*
** isnickopped()
**   Checks if the given nickname is opped on the channel.
**   Parameters:
**     nick - Nickname to check op state for
**   Returns:
**     1 if the nickname is opped, 0 if it is not, or if it is not a
**     user which MrsBot monitors.
**   PDL:
**     Locate the nickname in the internal store.  Return the opstate.
**     Simple, no?
*/
char isnickopped(char* nick)
{
  struct trackrec *tmpptr;

  tmpptr = locatenick(nick);
  if (tmpptr)
    return tmpptr->opped;
  else
    return 0;
}

/*
** onnick()
**   Called in response to a nick change server message.
**   Parameters:
**     oldnick - Old nickname being changed
**     newnick - New nickname user is changing to
**   Returns: void
**   PDL:
**     If it is MrsBot changing her nick, update the global variable
**     holding her current nick.  Otherwise, search the internal
**     store of channel users for a match with the old nick.  If one
**     is found, update the record to reflect the new nickname.
*/
void onnick(char* oldnick,char* newnick)
{
  struct trackrec *tmpptr;
  static int count = 0;
  static int nicktime;
  static char olduserhost[80];

  if (!strcmp(oldnick,mynick))
    strcpy(mynick,newnick); 

  /* SEARCH THE LIST OF MONITORED NICKS TO SEE IF ONE
     OF THEM CHANGED THEIR NICK. */
  tmpptr = locatenick(oldnick);
  if (tmpptr)
  {
    strcpy(tmpptr->nick,newnick);

	  if (!strcmp(olduserhost, tmpptr->userhost) && 
			 protecteduh(tmpptr->userhost) <= 0 )
	  {
		  if (time(NULL) - nicktime < 60) {
			 if (++count > 3) {
				  notice(mychannel, "\002NICK FLOOD DETECTED\002\n");
				  kick(mychannel, newnick);
				  count = 0;
				  nicktime = 0;
			  }
			}
		  else
		  {
			  nicktime = time(NULL);
			  count = 0;
			}
	  }
	  else
	  {
	     strcpy(olduserhost, tmpptr->userhost);
		  count = 0;
		  nicktime = 0;
     }
  }

}

/*
** whoopped()
**   Called when a nickname other than MrsBot is opped.
**   Parameters:
**     nick - Nickname that has been opped
**   Returns:
**     Userhost of person who was opped, or NULL if they are not being
**     monitored by MrsBot.
**   PDL:
**     Find the nick in MrsBot's internal user store.  Change their
**     op state to true.  Return the userhost from the record.
*/
char *whoopped(char* nick)
{
  struct trackrec *tmpptr;

  /* SEARCH THE LIST OF MONITORED NICKS TO SEE IF ONE HAS BEEN
     OPPED.  IF SO, MARK IT. */
  tmpptr = locatenick(nick);
  if (tmpptr) {
    tmpptr->opped = 1;
    return tmpptr->userhost;
    }
  return NULL;
}

/*
** whodeopped()
**   Called when a nickname other than MrsBot is de-opped.
**   Parameters:
**     nick - Nickname that has been deopped
**   Returns:
**     Userhost of person who was deopped, or NULL if they are not
**     being monitored by MrsBot.
**   PDL:
**     Find the nick in MrsBot's internal user store.  Change their
**     op state to false.  Return the userhost from the record.
*/
char *whodeopped(char* nick)
{
  struct trackrec *tmpptr;

  /* SEARCH THE LIST OF MONITORED NICKS TO SEE IF ONE HAS BEEN
     DE-OPPED.  IF SO, MARK IT. */
  tmpptr = locatenick(nick);
  if (tmpptr) {
    tmpptr->opped = 0;
    return tmpptr->userhost;
    }
  return NULL;
}
  
/*
** wholeft()
**   Called when someone has left the channel, either by leaving,
**    signing off, or being kicked off the channel.
**   Parameters:
**     nick - Nickname that left the channel
**     userhost - Output buffer to hold userhost that left
**   Returns: void
**   PDL:
**     If the first item of the internal list of channel users matches
**     the nickname, we got it easy.  Copy the userhost into the buffer,
**     move the top of the queue down one, and free up the unused record.
**     Otherwise, search the list for a matching nickname.  If one is
**     found, excise it from the list, copy the userhost, and free up
**     the record.  Note that we always set the ugly kludge key to zero
**     to indicate that the record has been deallocated.  If we don't
**     find a match, return a null string in userhost.
*/
void wholeft(char* nick,char* userhost)
{
  struct trackrec *tmpptr,*holdptr;

  /* SEARCH THE LIST OF MONITORED NICKS FOR A MATCH.
     IF SO, REMOVE THEM FROM THE LIST. */
  if (thoseweop)
    if (!strcmp(nick,thoseweop->nick)) {
      strcpy(userhost,thoseweop->userhost);
      holdptr = thoseweop;
      thoseweop = thoseweop->next;
      holdptr->Kludge_Key = 0;
      free(holdptr);
      --usersonchan;
      return;
      }
    else {
      tmpptr = thoseweop;
      while (tmpptr->next)
        if (!strcmp(nick,tmpptr->next->nick)) {
          holdptr = tmpptr->next;
          tmpptr->next = tmpptr->next->next;
          /* SET THE KLUDGE VALUE TO ZERO.  THIS ALLOWS ME TO BE LAZY AND
             NOT POP PENDING ALARMS FOR THIS PERSON OFF THE TIME QUEUE,
             SINCE THEY WILL BE REFERENCING A FREED OBJECT. */
          holdptr->Kludge_Key = 0;
          strcpy(userhost,holdptr->userhost);
          free(holdptr);
          --usersonchan;
          return;
          }
        else
          tmpptr = tmpptr->next;
      }
  *userhost = '\0';
}

/*
** alarmoff()
**   Called by Unix as the ALARM handler every time a timer expires.
**   Parameters: None
**   Returns: void
**   PDL:
**     Go thru the time ordered queue and locate all records that expire
**     the same second as the record at the top of the queue.  After this,
**     timequeue will be left pointing at the next record to be handled
**     AFTER this second, and tmpptr will be left pointing at all timers
**     to be handled by this alarm.  Go thru each of the alarms that are
**     to be handled this second and for each, do the following:  If the
**     request is for delayed ops, check the record for the user to be
**     opped.  If it has NOT been deallocated and the user has not yet
**     been opped, op him and update his user record to reflect this.  If
**     we are not opped, tell the poor soul that.  If the request is to
**     "miss Hendrix", check to see if he has returned to the channel.
**     If he has not, whine to the channel!  If the request is to turn
**     revenge kicking back on, update the spite global.  If the
**     request is to rejoin the channel, check if we have since been
**     able to join a channel.  If not, join the specified channel.  If
**     the request is to kill BabyBot, check if BabyBot has been inactive
**     for a minute or more.  If so, kill it.  If the request is to
**     change back to out original nickname, try it.  If it fails, the
**     nick collision routine will be called again.  Finally, if the
**     request is to ask a bot for ops, and we are not currently opped,
**     check for a helper bot and ask away.  After completing the action
**     for this timer record, free it.  Once all timers that expire this
**     second have been handled, calculate the amount of time to set the
**     alarm for for the item at the NEW top of the timer queue and set
**     up the alarm.
*/
void alarmoff()
{
  struct time_q_rec *tmpptr, *freeme;
  struct trackrec *convert;

  /* POP OFF ALL ALARMS THAT EXPIRE THIS SECOND */
  tmpptr = timequeue;
  while (timequeue->next)
    if (timequeue->next->expiretime == timequeue->expiretime)
      timequeue = timequeue->next;
    else
      break;
  timequeue = timequeue->next;

  /* GO THRU THE LIST OF EXPIRED ALARMS AND EXECUTE THE COMMANDS */
  while (tmpptr != timequeue) {
    switch (tmpptr->operation) {
      case DELAY_OP:
        convert = (struct trackrec *)tmpptr->oper_info;
        if (!convert->opped && convert->Kludge_Key == FREED_REC_KLUDGE) {
          if (opstate) {
            op (mychannel,convert->nick);
            convert->opped = 1;
	    }
          else
            notice (convert->nick,NOOPTOOP);
          }
        break;
      case DELAY_RESPITE:
	spite = 1;
	break;
      case DELAY_REJOIN:
	if (*mychannel == '\0') {
	  join(tmpptr->oper_info);
	  free(tmpptr->oper_info);
	  }
        break;
      case DELAY_SUBBOTDIE:
	 killbaby(tmpptr->oper_info); 
        break;
      case DELAY_RENICK:
        newnick(NICK);
	     break;
      case DELAY_GETOPS:
	if (!opstate)
	  askforops();
      default:
        break;
      }

    freeme = tmpptr;
    tmpptr = tmpptr->next;
    free(freeme);
    }

  /* SET A NEW ALARM FOR THE NEW TOP OF QUEUE */
/*  if (timequeue)
    alarm(timequeue->expiretime - time(NULL)); */
}

/*
** onpart()
**   Called on a server PART or SIGNOFF message. (Leaving channel)
**   Parameters:
**     nick - Nickname that left the channel
**   Returns: void
**   PDL:
**     Remove this user form the internal store of channel users.
**     Then check if the userhost that left is the bot owner..  If it is,
**     set up a timer to whine about how we miss him.
*/
void onpart(char* nick)
{
  char userhost[80];
  wholeft(nick,userhost);
}

/*
** timersignal()
**   Called to ensure that timer callbacks will not interfere
**    with partial PRIVMSG receipts.  This can cause seg faults.
**   Parameters:
**     signalvec - Pointer to array of callback addresses
**   Returns: void
**   PDL:
**     Scan through the list of callback addresses and ensure that
**     the signal is masked for each of these callbacks so as not to
**     cause seg faults during incomplete PRIVMSG receipt.
*/
void timersignal(char* signalvec)
{
  char sigmsk = MASKSIGS;

  while (*(++signalvec))
    sigmsk = ISMASKED(*signalvec);
}

/*
** resyncoplist()
**   Called when joining a channel to throw out the old internal store
**    of channel users and create a new one.
**   Parameters:
**     chan - Channel MrsBot is on
**   Returns: void
**   PDL:
**     Go thru each item of the old user store and free each record.
**     Then execute a WHO command.  The onwho() handler will take care
**     of rebuilding a new user store.
*/
void resyncoplist(char* chan)
{
  struct trackrec *holdptr;
  
  /* DELETE THE ENTIRE OP LIST AND INITIALIZE IT FROM A WHO */
  while (thoseweop) {
    holdptr = thoseweop;
    thoseweop = thoseweop->next;
    /* SET THE KLUDGE VALUE TO ZERO.  THIS ALLOWS ME TO BE LAZY AND
       NOT POP PENDING ALARMS FOR THIS PERSON OFF THE TIME QUEUE,
       SINCE THEY WILL BE REFERENCING A FREED OBJECT. */
    holdptr->Kludge_Key = 0;
    free(holdptr);
    }
  usersonchan = 0;
  who (chan);
}

/*
** onwho()
**   Handler for WHOREPLY server messages.  Builds MrsBot's internal channel
**    user store.
**   Parameters:
**     wholine - The test of the line from the WHOREPLY message.
**   Returns: void
**   PDL:
**     Parse out the userhost, nickname, channel (stays in wholine), and
**     user mode (e.g. the part that says "H@" on the who line).  Add this
**     user to the internal store if they are on our channel.
*/
void onwho(char* wholine)
{
  char *userhost, *nick, *mode, *tmp, *tmpchan;
  int isopped;

  userhost = strchr(wholine,' ');
  if (!userhost) return;
  *(userhost++) = '\0';
  nick = strchr(userhost,' ');
  if (!nick) return;
  *nick = '@';
  nick = strchr(userhost,' ');
  if (!nick) return;
  *(nick++) = '\0';
  nick = strchr(nick,' ');
  if (!nick) return;
  ++nick;
  mode = strchr(nick,' ');
  if (!mode) return;
  *(mode++) = '\0';
  tmp = strchr(mode,' ');
  if (!tmp) return;
  *tmp = '\0';
  if (*mychannel == ':') {
    tmpchan = mychannel;
    strcpy(mychannel, tmpchan+1);
  }
  if (!strcmp(mychannel,wholine)) {
    isopped = (strchr(mode,'@') != NULL);
    monitornick(nick,userhost,isopped);
    }
}

/*
** onendofwho()
**   Handles the server message indicating the end of WHO information
**    (numeric 315)
**   Parameters:
**     what - What we did the who on (either a channel name or username)
**   Returns: void
**   PDL:
**     This code has kind of gone away as onwho() has been changed from
**     handling public access ops, to ban beating, and now on to this.
*/
void onendofwho(char* what)
{
}

/*
** onnicktaken()
**   Called on a "Nick is already in use message" (numeric 433)
**   Parameters: None
**   Returns: void
**   PDL:
**     OK, some putz is stealing MrsBot's nickname.  Generate a random
**     nickname (MrsBot plus a number) and change to that nickname.
**     If we are not on a channel, join our default channel now.  This
**     is in case we had a nickname collision while signing on to IRC.
**     Finally, schedule an attempt to get back our normal nick.  ADDED:
**     do not switch nick if we already have a MrsBot<N> nick and MrsBot
**     is not available.
*/
void onnicktaken()
{
  char randnick[10];

  sprintf(randnick,"%s%1d", NAME, random() % 50);
  if (*mychannel == '\0') {
    newnick(randnick);
    strcpy(mynick,randnick);
    join(mychannel); 
    }
  else if (strncmp(mynick,NAME,strlen(NAME)))
    newnick(randnick);
  timednick(60);
}

/*
** int_userhost()
**   Called to obtain the userhost for a given nickname from MrsBot's
**    internal store of channel users.
**   Parameters:
**     nick - Nickname to search for
**   Returns:
**     Userhost for this nickname, or NULL if they are not in MrsBot's
**     internal store.
**   PDL:
**     Find the nick in MrsBot's internal user store.  Return the userhost
**     from the record.
*/
char *int_userhost(char* nick)
{
  struct trackrec *tmpptr;

  tmpptr = locatenick(nick);
  if (tmpptr)
    return tmpptr->userhost;
  return NULL;
}

/*
** randomuser()
**   Selects the nickname of a random user currently on the channel
**   Parameters:
**   Returns:
**     The nickname, of course. :)
**   PDL:
**     Select a random number between 0 and the number of users on the
**     channel.  Scan thru MrsBot's internal store until that many
**     users have been passed.  Return the nickname.  Easy.
*/
char *randomuser()
{
  int i;
  struct trackrec *tmpptr;

  tmpptr = thoseweop;
  for (i=random() % usersonchan;i>0;--i)
    if (tmpptr)
      tmpptr = tmpptr->next;
  if (tmpptr)
    return tmpptr->nick;
  else
    return NULL;
}
